<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package beleco
 */

 /**
 * Remove the child theme step.
 * This is for a Child theme.
 *
 * @since   0.1.0
 *
 * @return  $array  The merlin import steps.
 */
function tk_remove_child_step( $steps ) {
	unset( $steps['child'] );
	return $steps;
}
add_filter( get_template() . '_merlin_steps', 'tk_remove_child_step');

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
if ( ! function_exists( 'beleco_body_classes' ) ) :
	function beleco_body_classes( $classes ) {

		// Adds a class of tk-theme-frontend when viewing frontend.
		if ( !is_admin() ) {
			$classes[] = esc_attr( 'tk-theme-frontend' );
		}

		return $classes;
	}
	add_filter( 'body_class', 'beleco_body_classes' );
endif;

/**
 * Adds 'no-js' class.
 *
 * If we're missing JavaScript support, the HTML element will have a 'no-js' class.
 */
function beleco_no_js_class() {

	?>
	<script>document.documentElement.className = document.documentElement.className.replace( 'no-js', 'js' );</script>
	<?php

}
add_action( 'wp_head', 'beleco_no_js_class' );

/**
 * Disable woocommerce plugin activation redirect
 */
function tk_filter_woocommerce_enable_setup_wizard( $true ) {
	return false;
};
add_filter( 'woocommerce_enable_setup_wizard', 'tk_filter_woocommerce_enable_setup_wizard', 10, 1 );

/**
 * Filter the home page title from your demo content.
 * If your demo's home page title is "Home", you don't need this.
 */
if ( ! function_exists( 'beleco_merlin_content_home_page_title' ) ) :
	function beleco_merlin_content_home_page_title( $output ) {
		return 'Home';
	}
	add_filter( 'merlin_content_home_page_title', 'beleco_merlin_content_home_page_title' );
endif;


/**
 * Filter the blog page title from your demo content.
 * If your demo's blog page title is "Blog", you don't need this.
 */
if ( ! function_exists( 'beleco_merlin_content_blog_page_title' ) ) :
	function beleco_merlin_content_blog_page_title( $output ) {
		return 'Journal';
	}
add_filter( 'merlin_content_blog_page_title', 'beleco_merlin_content_blog_page_title' );
endif;


/**
 *  Woocommerce functions
 */
if ( ! function_exists( 'is_woocommerce_activated' ) ) {
	/**
	 * Enable gutenberg for woocommerce
	 */
	function tk_activate_gutenberg_product( $can_edit, $post_type ) {
		if ( 'product' == $post_type ) {
			$can_edit = true;
		}
		return $can_edit;
	}
	add_filter( 'use_block_editor_for_post_type', 'tk_activate_gutenberg_product', 10, 2 );

	/**
	 * Enable taxonomy fields for woocommerce with gutenberg on
	 */
	function tk_enable_taxonomy_rest( $args ) {
		$args['show_in_rest'] = true;
		return $args;
	}
	add_filter( 'woocommerce_taxonomy_args_product_cat', 'tk_enable_taxonomy_rest' );
	add_filter( 'woocommerce_taxonomy_args_product_tag', 'tk_enable_taxonomy_rest' );

	/**
	 * Remove description title in product tabs
	 */
	add_filter( 'woocommerce_product_description_heading', '__return_null' );

	/**
	 * Display product first gallery image next to thumbnail in woocommerce loop
	 */
	if ( ! function_exists( 'beleco_add_woo_gallery_image_to_archive' ) && !is_admin() ) :
		function beleco_add_woo_gallery_image_to_archive() {
			global $product;

			$attachment_ids = $product->get_gallery_image_ids();

			if ( ! empty( $attachment_ids ) ) {
				$image_link = wp_get_attachment_image_src( $attachment_ids[0], 'woocommerce_thumbnail')[0];
				$image_alt = get_post_meta( $attachment_ids[0], '_wp_attachment_image_alt', true );
				if (!$image_alt) {
					$image_alt = esc_html__( 'Hover Product Image', 'beleco' );
				}
				printf( '<img src="%1s" alt="%2s">', esc_url( $image_link ), esc_textarea( $image_alt ) );
			}
		}
		add_action( 'woocommerce_before_shop_loop_item_title', 'beleco_add_woo_gallery_image_to_archive', 11 );
	endif;

	/**
	 * Display product first gallery image next to thumbnail in product block
	 */

	if ( !function_exists( 'beleco_add_woo_gallery_image_to_product_block' ) && !is_admin() ) :
		function beleco_add_woo_gallery_image_to_product_block( $html, $data, $product ) {
			$attachment_ids = $product->get_gallery_image_ids();
			$galleryImg = '';

			if ( ! empty( $attachment_ids ) ) {
				$image_link = wp_get_attachment_image_src( $attachment_ids[0], 'woocommerce_thumbnail')[0];
				$image_alt = get_post_meta( $attachment_ids[0], '_wp_attachment_image_alt', true );

				if (!$image_alt) {
					$image_alt = esc_html__( 'Hover Product Image', 'beleco' );
				}

				if ( $image_link ) {
					$galleryImg = '<img src="' . esc_attr( $image_link ) . '" alt="' . esc_attr( $image_alt ) . '">';
				}
			}

			$output = '<li class="wc-block-grid__product">
				<a href="' . $data->permalink . '" class="wc-block-grid__product-link">
					' . $data->image . '
					' . $galleryImg . '
					' . $data->title . '
				</a>
				' . $data->badge . '
				' . $data->price . '
				' . $data->rating . '
				' . $data->button . '
			</li>';

			return $output;
		}
		add_filter( 'woocommerce_blocks_product_grid_item_html', 'beleco_add_woo_gallery_image_to_product_block', 10, 3);
	endif;
}

/**
 * Trim dots at the end of excerpt
 *
 */
if ( ! function_exists( 'tk_trim_excerpt' ) ) :
	function tk_trim_excerpt( $more ) {
		return '';
	}
	add_filter('excerpt_more', 'tk_trim_excerpt');
endif;

/**
 * Add product categories on each product in product loop
 *
 */
if ( ! function_exists( 'tk_categories_on_product_archives' ) ) :
	function tk_categories_on_product_archives() {
		global $product;

		echo '<div class="posted_in">' . wp_kses( wc_get_product_category_list( $product->get_id(), ' ' ), 'post' ) . '</div>';

	}
	add_action( 'woocommerce_before_shop_loop_item_title', 'tk_categories_on_product_archives', 20 );
endif;

/**
 * Add excerpt to product archives, limited to 25 words
 *
 */
if ( ! function_exists( 'tk_excerpt_on_product_archives' ) ) :
	function tk_excerpt_on_product_archives() {

		echo '<p class="product-excerpt">' . esc_html__( wp_trim_words( get_the_excerpt(), 25 ) ) . '</p>';

	}
	add_action( 'woocommerce_shop_loop_item_title', 'tk_excerpt_on_product_archives', 40 );
endif;


/**
 * Add short description to WooCommerce product blocks and limt to 25 words
 */
if ( ! function_exists( 'tk_add_short_desc_woocommerce_blocks_product_grid' ) ) :
	function tk_add_short_desc_woocommerce_blocks_product_grid( $content, $data, $product ) {
		$short_description_string = strip_tags( $product->get_short_description() );
		$words = explode( ' ', $short_description_string );
		$limit_words = trim(implode( ' ', array_slice( $words, 0, 25 ) ) );
		if ( strlen( $limit_words ) < strlen( $short_description_string ) ) {
			$limit_words .= '...';
		}
		$short_description = '<div class="wc-block-grid__product-short-description product-excerpt">' . $limit_words . '</div>';

		// This will inject the short description after the first link (assumed: the default product link).
		$after_link_pos = strpos( $content, '</a>' );
		$before = substr($content, 0, $after_link_pos + 4);
		$after = substr($content, $after_link_pos + 4);

		$content = $before . $short_description . $after;

		return $content;
	}
	add_filter('woocommerce_blocks_product_grid_item_html', 'tk_add_short_desc_woocommerce_blocks_product_grid', 10 , 3);
endif;

/**
 * Override Woocommerce default pagination arrows
 *
 */
if ( ! function_exists( 'tk_override_pagination_args' ) ) :
	function tk_override_pagination_args( $args ) {
		$args['prev_text'] = __( 'Prev', 'beleco' );
		$args['next_text'] = __( 'Next', 'beleco' );
		return $args;
	}
	add_filter( 'woocommerce_pagination_args' , 'tk_override_pagination_args' );
endif;

/**
 * Override Woocommerce gallery thumbnail size
 *
 */
if ( ! function_exists( 'tk_woocommerce_gallery_thumbnail' ) ) :
	function tk_woocommerce_gallery_thumbnail( $size ) {
		return array(
			'width' => 122,
			'height' => 0,
			'crop' => 0
		);
	}
	add_filter( 'woocommerce_get_image_size_gallery_thumbnail' , 'tk_woocommerce_gallery_thumbnail' );
endif;

/**
 * Remove woocommerce breadcrumbs
 *
 */
remove_action('woocommerce_before_main_content', 'woocommerce_breadcrumb', 20, 0);

/**
 * Add link to product title on Woocommerce archive pages
 *
 */
if ( ! function_exists( 'tk_change_products_title' ) ) :
	remove_action( 'woocommerce_shop_loop_item_title', 'woocommerce_template_loop_product_title', 10 );
	function tk_change_products_title() {
		echo '<h2 class="' . esc_attr( apply_filters( 'woocommerce_product_loop_title_classes', 'woocommerce-loop-product__title' ) ) . '"><a href="' . esc_attr( get_the_permalink() ) . '">' . esc_html( get_the_title() ) . '</a></h2>';
	}
	add_action( 'woocommerce_shop_loop_item_title', 'tk_change_products_title', 10 );
endif;
