<?php
/**
 * Custom functions that act independently of the theme templates.
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package Goodz
 */



/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function goodz_body_classes( $classes ) {
	// Get global layout setting
	$global_layout = get_theme_mod( 'archive_layout_setting', 'boxed' );
	$single_layout = get_theme_mod( 'single_layout_setting', 'boxed' );
	$sticky_header = get_theme_mod( 'sticky_header_setting', 0 );
	$slider_width  = get_theme_mod( 'featured_slider_width', 0 );
    $slider_enable = get_theme_mod( 'featured_slider_enable', 0 );
    $slider_autoplay = get_theme_mod( 'featured_slider_autoplay', 1 );
    $front_slider  = get_theme_mod( 'front_page_slider_enable', 1 );
    $front_slider_autoplay = get_theme_mod( 'front_page_slider_autoplay', 1 );

    if ( goodz_is_woocommerce_activated() && is_woocommerce() ) {

        return $classes;

    } else {

    	if ( is_single() ) {
        	$classes[] = $single_layout . '-single';
        } else {
        	if ( ! is_page() || is_page_template( 'templates/portfolio-page.php' ) ) {
    			$classes[] = $global_layout . '-blog';
    		}
        }

    	if ( is_home() ) {
            if ( $slider_enable && 'yes' == $slider_width ) {
    			$classes[] = 'featured-slider-fullwidth';
    		}

            if ( $slider_enable && $slider_autoplay ) {
                $classes[] = 'featured-slider-autoplay';
            }
    	}

        if ( is_page_template( 'templates/template-front-page.php' ) ) {

            if ( $front_slider ) {
                $classes[] = 'home-slider-enabled';

                if ( $front_slider_autoplay ) {
                    $classes[] = 'home-slider-autoplay';
                }
            }

        }

    	// Adds a class of group-blog to blogs with more than 1 published author.
    	if ( is_multi_author() ) {
    		$classes[] = 'group-blog';
        }

        // Adds a class of tk-theme-frontend when viewing frontend.
        if ( !is_admin() ) {
            $classes[] = 'tk-theme-frontend';
        }

        return $classes;

    }

}
add_filter( 'body_class', 'goodz_body_classes' );

/**
 * Filter content column classes
 *
 * @since goodz 1.0
 */
function goodz_content_cols() {

    if ( goodz_is_woocommerce_activated() ) {
        if ( is_cart() || is_checkout() || goodz_is_page( 'woocommerce-account' ) || goodz_is_page( 'woocommerce-wishlist' ) ) {
            return;
        }
    }

	// Primary container classes
	$cols = 'col-lg-12';

	// Get global layout setting
	$global_layout = get_theme_mod( 'archive_layout_setting', 'boxed' );

	if ( is_active_sidebar( 'sidebar-1' ) ) {

		if ( 'fullwidth' == $global_layout && ( is_archive() || is_home() ) ) {
			$cols = 'col-lg-10 col-md-9 has-sidebar';
		}
		else {
			$cols = 'col-md-9 has-sidebar';
		}
	}
    else {
        $cols = 'col-sm-12';
    }

	if ( is_single() ) {
		// Container classes relevant to sidebar
		$cols = 'no-sidebar';

		if ( is_active_sidebar( 'sidebar-1' ) ) {
		    $cols = 'has-sidebar';
		}
	}

	echo esc_attr( $cols );
}

/**
 * Check if page is WooCommerce my account
 */
function goodz_is_page( $class ) {

    $classes = get_body_class();

    // Check if class exists in body tag
    if ( is_array( $classes ) ) {

        if ( in_array( $class, $classes ) ){
            return true;

        } else {

            return false;

        }

    } else {

        return false;

    }

}

/**
 * Filter sidebar column classes
 *
 * @since goodz 1.0
 */
function goodz_sidebar_cols() {

	// Get global layout setting
	$global_layout = get_theme_mod( 'archive_layout_setting', 'boxed' );

	if ( is_active_sidebar( 'sidebar-1' ) && ( goodz_is_woocommerce_activated() && !is_cart() && !is_checkout() ) && !goodz_is_page( 'woocommerce-account' ) && !goodz_is_page( 'woocommerce-wishlist' ) ) {

        if ( 'fullwidth' == $global_layout && ( is_archive() || is_home() ) ) {
            $cols = 'col-lg-2 col-md-3';
        }
        else {
            $cols = 'col-md-3';
        }

	}
	else {
		$cols = 'col-md-3';
	}

	echo esc_attr( $cols );
}

/**
 * Filter post_class() additional classes
 *
 * @since goodz 1.0
 */
function goodz_post_classes( $classes, $class, $post_id ) {
	// Get global layout setting
	$global_layout     = get_theme_mod( 'archive_layout_setting', 'boxed' );
	$two_column_layout = get_theme_mod( 'two_columns_layout_setting', 0 );

	if ( ! is_single() ) :

		// If global layout is set to boxed
		if ( 'boxed' == $global_layout && 'product' != get_post_type() ) {
		    if ( is_sticky() ) {
		    	if ( $two_column_layout ) {
		    		$classes[] = 'col-sm-6';
		    	}
		    	else {
		    		if ( is_active_sidebar( 'sidebar-1' ) ) {
		    			$classes[] = 'col-sm-12';
		    		}
		    		else {
		    			$classes[] = 'col-lg-8 col-sm-12';
		    		}
		    	}
			}
			else {
				if ( $two_column_layout ) {
					$classes[] = 'col-sm-6';
				}
				else {
					if ( is_active_sidebar( 'sidebar-1' ) ) {
						$classes[] = 'col-sm-6';
					}
					else {
						$classes[] = 'col-lg-4 col-sm-6';
					}
				}
			}
		}
		else if ( 'fullwidth' == $global_layout && 'product' != get_post_type() ) {
			if ( is_sticky() ) {
				if ( $two_column_layout ) {
					$classes[] = 'col-sm-6';
				}
				else {
					if ( is_active_sidebar( 'sidebar-1' ) ) {
						$classes[] = 'col-lg-8 col-sm-12';
					}
					else {
						$classes[] = 'col-lg-6 col-sm-12';
					}
				}
			}
			else {
				if ( $two_column_layout ) {
					$classes[] = 'col-sm-6';
				}
				else {
					if ( is_active_sidebar( 'sidebar-1' ) ) {
                        $classes[] = 'col-lg-4 col-sm-6';
					}
					else {
                        $classes[] = 'col-lg-3 col-sm-6';
					}
				}
			}
		}

	endif;

	return $classes;

}
add_filter( 'post_class', 'goodz_post_classes', 10, 3 );

/**
 * Widget tag cloud font size update
 *
 * @since  goodz 1.0
 */
function goodz_widget_tag_cloud_args( $args ) {
	$args['largest']  = 14;
	$args['smallest'] = 14;
	$args['unit']     = 'px';
	return $args;
}
add_filter( 'widget_tag_cloud_args', 'goodz_widget_tag_cloud_args' );

/**
 * Check for embed content in post and extract
 *
 * @since goodz 1.0
 */
function goodz_get_embed_media() {
    $content = get_the_content();
    $embeds  = get_media_embedded_in_content( $content );

    if ( !empty( $embeds ) ) {
        //check what is the first embed containg video tag, youtube or vimeo
        foreach( $embeds as $embed ) {
            if ( strpos( $embed, 'video' ) || strpos( $embed, 'youtube' ) || strpos( $embed, 'vimeo' ) ) {
                return $embed;
            }
        }
    } else {
        //No video embedded found
        return false;
    }
}

/**
 * Add Read More to post excerpt
 *
 * @since  Goodz 1.0
 */
function new_excerpt_more( $excerpt ) {
	return $excerpt .' <a class="read-more" href="' . get_permalink( get_the_ID() ) . '">' . esc_html__( 'Read more', 'goodz' ) . '</a>';
}
add_filter( 'the_excerpt', 'new_excerpt_more' );

/**
 * Extract image from video
 */
function goodz_get_video_image( $url, $post_ID, $img_quality ) {

    require_once( ABSPATH . "wp-admin" . '/includes/image.php' );
    require_once( ABSPATH . "wp-admin" . '/includes/file.php' );
    require_once( ABSPATH . "wp-admin" . '/includes/media.php' );

    if ( !empty( $url ) ) {
        $key_str1    = 'youtube';
        $key_str2    = 'vimeo';
        $pos_youtube = strpos( $url, $key_str1 );
        $pos_vimeo   = strpos( $url, $key_str2 );

        if ( !empty( $pos_youtube ) ) {
            $url      = str_replace( 'watch?v=', '', $url );
            $url      = explode( '&', $url );
            $url      = $url[0];
            $protocol = substr( $url, 0, 5 );

            if ( $protocol == "http:" ) {
                $url      = str_replace( 'http://www.youtube.com/', '', $url );
                $protocol_new = "http://";
            }
            if ( $protocol == "https" ) {
                $url      = str_replace( 'https://www.youtube.com/', '', $url );
                $protocol_new = "https://";
            }

            if ( empty( $img_quality ) ) {
                $img_quality = 2;
            }

            $video_image_url = $protocol_new . 'img.youtube.com/vi/'. $url . '/hqdefault.jpg';

            if ( !has_post_thumbnail( $post_ID ) ) {
                $url = $video_image_url;
                $tmp = download_url( $url );

                if( is_wp_error( $tmp ) ){
                    // download failed, handle error
                }

                $post_id    = $post_ID;
                $desc       = get_the_title();
                $file_array = array();

                // Set variables for storage
                // fix file filename for query strings
                preg_match( '/[^\?]+\.(jpg|jpe|jpeg|gif|png)/i', $url, $matches );
                $file_array['name']     = basename( $matches[0] );
                $file_array['tmp_name'] = $tmp;

                // If error storing temporarily, unlink
                if ( is_wp_error( $tmp ) ) {
                    @unlink( $file_array['tmp_name'] );
                    $file_array['tmp_name'] = '';
                }

                // do the validation and storage stuff
                $id = media_handle_sideload( $file_array, $post_id, $desc );

                // If error storing permanently, unlink
                if ( is_wp_error( $id ) ) {
                    @unlink( $file_array['tmp_name'] );
                    return $id;
                }

                set_post_thumbnail( $post_ID, $id );

            }

            $video_image = get_the_post_thumbnail( $post_ID, 'goodz-video-featured-image' );

        }
        elseif ( !empty( $pos_vimeo ) ) {

            $urlParts = explode( "/", parse_url( $url, PHP_URL_PATH ) );
            $videoId  = (int) $urlParts[count($urlParts)-1 ];
            $data     = wp_remote_get( "http://vimeo.com/api/v2/video/" . $videoId . ".json" );

            if ( !is_wp_error( $data ) && is_array( $data ) ) {
                $data  = wp_remote_get( "http://vimeo.com/api/v2/video/" . $videoId . ".json" );
                $data  = json_decode( wp_remote_retrieve_body( $data ), true );
                $image = $data[0]['thumbnail_large'];

                if ( !has_post_thumbnail( $post_ID ) ) {
                    $url = $image;
                    $tmp = download_url( $url );

                    if( is_wp_error( $tmp ) ){
                        // download failed, handle error
                    }

                    $post_id    = $post_ID;
                    $desc       = get_the_title();
                    $file_array = array();

                    // Set variables for storage
                    // fix file filename for query strings
                    preg_match( '/[^\?]+\.(jpg|jpe|jpeg|gif|png)/i', $url, $matches );
                    $file_array['name']     = basename( $matches[0] );
                    $file_array['tmp_name'] = $tmp;

                    // If error storing temporarily, unlink
                    if ( is_wp_error( $tmp ) ) {
                        @unlink( $file_array['tmp_name'] );
                        $file_array['tmp_name'] = '';
                    }

                    // do the validation and storage stuff
                    $id = media_handle_sideload( $file_array, $post_id, $desc );

                    // If error storing permanently, unlink
                    if ( is_wp_error( $id ) ) {
                        @unlink( $file_array['tmp_name'] );
                        return $id;
                    }

                    set_post_thumbnail( $post_ID, $id );

                }

                $video_image = get_the_post_thumbnail( $post_ID, 'goodz-video-featured-image' );

            }
        }
        else {

            $video_image = esc_html__( 'Video only allowes vimeo and youtube!', 'goodz' );
        }

        return $video_image;
    }
}

/**
 * Remove website vrom comment form
 *
 * @since Goodz 1.0
 */
function goodz_disable_comment_url( $fields ) {
    unset( $fields['url'] );
    return $fields;
}
add_filter( 'comment_form_default_fields', 'goodz_disable_comment_url' );

/**
 * Set maximum width of images for responsive images feature
 *
 * @since Goodz 1.0
 */
function goodz_m_max_srcset_image_width( $max_width, $size_array ) {
    $width = $size_array[0];

    if ( $width > 600 ) {
        $max_width = 2048;
    }

    return $max_width;
}
add_filter( 'max_srcset_image_width', 'goodz_m_max_srcset_image_width', 10, 2 );

/**
 * If is Front Page Template
 *
 * @since  Goodz 1.0
 */
function goodz_is_front_template() {
    return is_page_template( 'templates/template-front-page.php' );
}

/**
 * Check if WooCommerce is activated
 *
 * @since Goodz 1.0
 */
if ( ! function_exists( 'goodz_is_woocommerce_activated' ) ) {
    function goodz_is_woocommerce_activated() {
        return class_exists( 'woocommerce' ) ? true : false;
    }
}

/**
 * Exclude Category from Blog if slider and category enabled
 *
 * Alters main query to get wanted results
 */
function goodz_exclude_category_from_blog( $query ) {

    // Get Featured Slider settings
    $featured_slider     = get_theme_mod( 'featured_slider_enable', 0 );
    $featured_category   = get_theme_mod( 'featured_category_select', 'default' );
    $featured_slider_cat = get_theme_mod( 'featured_slider_cat_exclude', 0 );

    if ( $featured_slider && $featured_slider_cat ) :

        if ( $query->is_main_query() && $query->is_home() ) {

            if ( 'default' != $featured_category ) {

                $category_exclude    = get_category_by_slug( $featured_category );
                $category_exclude_id = $category_exclude->term_id;

                $query->set( 'category__not_in', array( $category_exclude_id ) );
            }

        }

    endif;

    if ( $query->is_main_query() && $query->is_home() ) {
        if ( $query->is_paged() ) {
            $query->set( 'ignore_sticky_posts', 1 );
        }
    }

}
add_action( 'pre_get_posts', 'goodz_exclude_category_from_blog' );

/**
 * Do shortcode function instead calling do_shortcode
 *
 */
function goodz_do_shortcode_function( $tag, array $atts = array(), $content = null ) {

	 global $shortcode_tags;

	 if ( ! isset( $shortcode_tags[ $tag ] ) )
			 return false;

	 return call_user_func( $shortcode_tags[ $tag ], $atts, $content, $tag );
}
